function [ ] = DQSpline_Plot( dt, Fsample, n )
%Plot the derivative function for quadratic spline
%   dt is Delaunauy Triangulation and Fsample should be formatted like the
%   output from Q_Fsample. Like QSpline_Plot, n is the number of sample
%   points taken per triangle.

numtri = size(dt.Triangulation,1);


%Interpolate the different splines indexed to the different triangles.
C = zeros(numtri, 6);

for i=1:numtri
    %The rows store the spline coefficients on {1,x,y,xy,x^2,y^2} basis for
    %triangle i
    C(i,:) = Q_Terp([ Fsample(i,:,1)' Fsample(i,:,2)' ],Fsample(i,:,3)');
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%We need to pick the test points which we'll sample the spline at for
%use in Matlab's plot function.  The idea here is that we'll seed the std
%element to some precision, then map these points into each triangle via
%Std_to_Tri. 'n' is the key parameter, being the number of points taken
%along e1, and one less each line above until the vertex e2 at which there
%is just the 1.

%Grid the standard element
s = 0:1/(n-1):1;
t = 0:1/(n-1):1;
[S T] = meshgrid(s,t); %This gives us grids for a square but need to only 
                       %be our triangle!
                       
S = 1 - S;          %For those vertices of square grid not in <0,e1,e2> triangle
S = triu(S);        %pad those with a repeat of the origin. This ensures
T = triu(T);        %we only mesh the triangle. NOTE: The upper triangular part
                    %with the diagonal is carrying actual info on triangle.
                    %The STRICTLY lower part is padded as 0 in both x and y

%We now need to map these standard points into each triangle within our
%actual triangulation.

X = zeros(n*numtri,n); %To each triangle is n x n grid that is 
Y = zeros(n*numtri,n); %'padded' with the repeat vertex to make triangular
Zx = zeros(n*numtri,n); %Where we'll store the x-deriv values
Zy = zeros(n*numtri,n); %Where we'll store the y-deriv values
I = zeros(n,n)+1; %We'll use to make sure true padded and no numerical roundoff error
I = triu(I); %Is a strictly lower triangle matrix with 1's.

h = waitbar(0,'How Many Triangles Have Been Evaluated');
for i=1:numtri  %Indexes which triangle we're at.
    Triangle = zeros(3,2);
    Triangle(:,1) = arrayfun(@(l) dt.X(l,1), dt.Triangulation(i,:)'); %Extract triangle vertices: x-coordinates
    Triangle(:,2) = arrayfun(@(l) dt.X(l,2), dt.Triangulation(i,:)'); %Extract triangle vertices: y-coordinates
    
    X((i-1)*n+1:i*n,:) = arrayfun(@(u,v) Std_to_Tri(Triangle,[[u v]])*[[1 0]]', S, T); %Map in the points: x-coordinates
    Y((i-1)*n+1:i*n,:) = arrayfun(@(u,v) Std_to_Tri(Triangle,[[u v]])*[[0 1]]', S, T); %Map in the points: y-coordinates
    
    B = triu(X((i-1)*n+1:i*n,:)) - diag(diag(X((i-1)*n+1:i*n,:))); %Instead of padding by a single point which made surf look buggy,
    X((i-1)*n+1:i*n,:) = X((i-1)*n+1:i*n,:).*I+B';                 %We double all off diagonal points, we'll see if better
    
    B = triu(Y((i-1)*n+1:i*n,:)) - diag(diag(Y((i-1)*n+1:i*n,:))); % 
    Y((i-1)*n+1:i*n,:) = Y((i-1)*n+1:i*n,:).*I+B';
    
    %With the points in the ith triangle known, we can go ahead and compute
    %the derivative spline values
    
    Zx((i-1)*n+1:i*n,:) = arrayfun(@(u,v) C(i,2) + 2*C(i,5).*u + C(i,4).*v, X((i-1)*n+1:i*n,:), Y((i-1)*n+1:i*n,:)); %Evaluate the spline deriv using C coefficients.
    Zy((i-1)*n+1:i*n,:) = arrayfun(@(u,v) C(i,3) + C(i,4).*u + 2*C(i,6).*v, X((i-1)*n+1:i*n,:), Y((i-1)*n+1:i*n,:)); %Evaluate the spline deriv using C coefficients.
    
    Bx = triu(Zx((i-1)*n+1:i*n,:)) - diag(diag(Zx((i-1)*n+1:i*n,:))); %again how we duplicate to make surf behave better
    Zx((i-1)*n+1:i*n,:) = Zx((i-1)*n+1:i*n,:).*I+Bx';
    
    By = triu(Zy((i-1)*n+1:i*n,:)) - diag(diag(Zy((i-1)*n+1:i*n,:))); %again how we duplicate to make surf behave better
    Zy((i-1)*n+1:i*n,:) = Zy((i-1)*n+1:i*n,:).*I+By';
    waitbar(i/numtri,h)

end
close(h)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%Render the x deriv in a new figure diagram
figure
hold
for i=1:numtri
surfl(X((i-1)*n+1:i*n,:),Y((i-1)*n+1:i*n,:),Zx((i-1)*n+1:i*n,:),'light') %We plot each individual triangle so that Matlab doesn't try to connect what shouldnt connect
end
hold

%Render the y deriv in a new figure diagram
figure
hold
for i=1:numtri
surfl(X((i-1)*n+1:i*n,:),Y((i-1)*n+1:i*n,:),Zy((i-1)*n+1:i*n,:),'light') %We plot each individual triangle so that Matlab doesn't try to connect what shouldnt connect
end
hold

end

